#!/usr/bin/env bash
#========================================================
# 脚本名称： mgr_mode_switching.sh
# 版    本： v0.3
# 作    者： WHao
# 创建日期： 2025‑05‑17
# 用    途： MySQL Group Replication 集群 单主↔多主 切换
#========================================================
set -euo pipefail

#--------------------------------------------------------
# 1) 变量配置
#--------------------------------------------------------
source /tmp/mgr_cluster_config.env          # 包含 MASTER_IP / SLAVE_IPS
IFS=' ' read -r -a slave_ips <<<"$SLAVE_IPS"

MYSQL_PWD=${MYSQL_PWD:-"123456"}            # 也可在 shell 里 export 覆盖
MYSQL_OPTS="-uroot -p'$MYSQL_PWD' -N -s"

RPL_USER='rpl_user'
RPL_PWD='QWERqwer'

REMOTE_SQL_DIR="/root"                      # 远端 SQL 默认落脚点

#--------------------------------------------------------
# 2) 生成 SQL 文件
#--------------------------------------------------------
make_sql_files() {
cat > stop_single_mode.sql <<'EOS'
STOP GROUP_REPLICATION;
SET GLOBAL group_replication_single_primary_mode = OFF;
SET GLOBAL group_replication_enforce_update_everywhere_checks = ON;
EOS

cat > start_master_multi_mode.sql <<EOS
-- 确保参数为多主
SET GLOBAL group_replication_single_primary_mode = OFF;
SET GLOBAL group_replication_enforce_update_everywhere_checks = ON;
SET @@SESSION.sql_log_bin = 0;

SET GLOBAL group_replication_bootstrap_group = ON;
START GROUP_REPLICATION USER='$RPL_USER', PASSWORD='$RPL_PWD';
SET GLOBAL group_replication_bootstrap_group = OFF;
EOS

cat > start_slave_multi_mode.sql <<EOS
START GROUP_REPLICATION USER='$RPL_USER', PASSWORD='$RPL_PWD';
EOS

cat > stop_multi_mode.sql <<'EOS'
STOP GROUP_REPLICATION;
SET GLOBAL group_replication_enforce_update_everywhere_checks = OFF;
SET GLOBAL group_replication_single_primary_mode = ON;
EOS

cat > start_master_single_mode.sql <<EOS
SET GLOBAL group_replication_bootstrap_group = ON;
SET GLOBAL group_replication_single_primary_mode = ON;
START GROUP_REPLICATION USER='$RPL_USER', PASSWORD='$RPL_PWD';
SET GLOBAL group_replication_bootstrap_group = OFF;
EOS

cp start_slave_multi_mode.sql start_slave_single_mode.sql
}

#--------------------------------------------------------
# 3) 把 SQL 传到所有节点
#--------------------------------------------------------
deploy_sql_files() {
  for ip in "${slave_ips[@]}"; do
    scp *.sql root@"$ip":"$REMOTE_SQL_DIR"/
  done
}

#--------------------------------------------------------
# 4) 切换函数
#--------------------------------------------------------
enable_multi_master() {
  for ip in "${slave_ips[@]}"; do
    ssh root@"$ip" "mysql $MYSQL_OPTS < $REMOTE_SQL_DIR/stop_single_mode.sql"
  done

  ssh root@"$MASTER_IP" \
      "mysql $MYSQL_OPTS < $REMOTE_SQL_DIR/start_master_multi_mode.sql"

  for ip in "${slave_ips[@]:1}"; do
    ssh root@"$ip" "mysql $MYSQL_OPTS < $REMOTE_SQL_DIR/start_slave_multi_mode.sql"
  done
}

enable_single_primary() {
  for ip in "${slave_ips[@]}"; do
    ssh root@"$ip" "mysql $MYSQL_OPTS < $REMOTE_SQL_DIR/stop_multi_mode.sql"
  done

  ssh root@"$MASTER_IP" \
      "mysql $MYSQL_OPTS < $REMOTE_SQL_DIR/start_master_single_mode.sql"

  for ip in "${slave_ips[@]:1}"; do
    ssh root@"$ip" "mysql $MYSQL_OPTS < $REMOTE_SQL_DIR/start_slave_single_mode.sql"
  done
}



#--------------------------------------------------------
# 获取真实模式 & 主节点
#--------------------------------------------------------
get_cluster_mode() {
  # 默认值，防止 set -u 下未赋
  MODE="Unknown"
  MASTER_NODE="N/A"

  # 通过 ssh 执行 mysql；SQL 用 Here‑Doc 送入，省去所有转义
  result=$(ssh -o ConnectTimeout=3 -T root@"$MASTER_IP" bash <<'REMOTE'
mysql -uroot -p123456 -N -s <<'SQL'
SELECT
  COALESCE(SUM(MEMBER_ROLE='PRIMARY'),0)  AS primary_cnt,
  COALESCE(MAX(CASE WHEN MEMBER_ROLE='PRIMARY' THEN MEMBER_HOST END),'') AS primary_ip
FROM performance_schema.replication_group_members
WHERE MEMBER_STATE IN ('ONLINE','RECOVERING','WAITING');
SQL
REMOTE
  ) || true   # 任何 ssh / mysql 失败都不让脚本退出

  # 解析结果——期望格式：<cnt>\t<ip>
  IFS=$'\t' read -r primary_cnt primary_ip <<<"$result"

  primary_cnt=${primary_cnt:-0}
  primary_ip=${primary_ip:-}

  if [[ $primary_cnt -eq 1 ]]; then
      MODE="Single‑Primary"
      MASTER_NODE=$primary_ip
  else
      MODE="Multi‑Primary"
      MASTER_NODE="所有节点均为 Primary"
  fi
}







#--------------------------------------------------------
# 6) 主程序
#--------------------------------------------------------
### 首次生成并分发 SQL
make_sql_files
deploy_sql_files

### 交互循环
while true; do
  get_cluster_mode
  clear
  echo -e "\e[31m==============  MGR 集群模式切换  ==============\e[0m"
  echo -e "\e[34m 当前模式: $MODE      Primary: $MASTER_NODE \e[0m"
  echo -e "\e[31m==============================================\e[0m"
  echo -e "\e[34m 1) 切到单主模式     2) 切到多主模式     q) 退出 \e[0m"
  echo -e "\e[31m==============================================\e[0m"
  read -rp "请选择 (1 / 2 / q): " choice

  case $choice in
    1)
      if [[ $MODE == "Single‑Primary" ]]; then
        echo "当前已是单主模式，无需切换"; sleep 2
      else
        echo ">>> 正在切换到单主模式..."
        enable_single_primary
        echo ">>> 成功切换为单主模式"; sleep 2
      fi
      ;;
    2)
      if [[ $MODE == "Multi‑Primary" ]]; then
        echo "当前已是多主模式，无需切换"; sleep 2
      else
        echo ">>> 正在切换到多主模式..."
        enable_multi_master
        echo ">>> 成功切换为多主模式"; sleep 2
      fi
      ;;
    q|Q)
      echo "Bye."
      exit 0
      ;;
    *)
      echo "无效输入"; sleep 1 ;;
  esac
done

